// LSW_Test64.cpp : Defines the entry point for the console application.
//
// RD 11/2014 -- 64 bit demo program
//
// V.9

#include "stdafx.h"
using namespace std;
#include <stdio.h>
#include <stdint.h>         
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include "vnx_lsw_api.h"

// ---------- Demonstration Program ------------------------------------------------------
//
// This program shows how to call the RF switch functions
//
// It is not intended as an example of appropriate user interface design, or the error handling
// that would be present in a production program.
// Several examples of testing for error codes returned by DLL function calls are included, but for simplicity
// many example calls do not test for error cases that might be returned.
//
// Note that the example code gets the state of the switch hardware before the commands are sent to it, running
// the program again will display the settings of the device which resulted from the last run of the program.

// ------------------------------- Equates -----------------------------------------------
#define SWPAT_REPEAT		0x02	// MASK: bit = 1 for continuous sweeping
#define SWPAT_ONETIME		0x01	// MASK: bit = 1 for single sweep


// ------------------------------- Allocations -------------------------------------------

static DEVID MyDevices[MAXDEVICES];				// I have statically allocated this array for convenience
												// It holds a list of device IDs for the connected devices
												// They are stored starting at MyDevices[0]

static char MyDeviceName[MAX_MODELNAME];		// this is a single byte char array for testing the ASCII name function
static wchar_t MyDeviceNameW[MAX_MODELNAME];	// this is a WCHAR array for testing the Unicode name function

static wchar_t errmsg[32];						// For the status->string converter
static char cModelName[32];						// buffer for the model name

static bool gbWantOneDevice = FALSE;
static int DevNum = 0;							// the device we should work with - index is UI version - 1.


// --------------------------- Variables -------------------------------------------------


static long SerialNumber = 0;			// used to hold the serial number for the get serial number command
static int Switch_Select = 0;			// used to hold the switch number to select
static int Switch_port = 0;			    // used to hold the switch port to select
static int Set_External_Control = 0;	// used to choose internal or external switch control signals
static int Use_Pulse_Mode = 0;			// used to enable pulsed mode operation if 1
static int Start_Pattern = 0;			// used to start the pattern if 1

static float ActivePulseTime = 1.0;		// default active pulse time is 1 second
static float InactivePulseTime = 1.0;	// default inactive pulse time is also 1 second

static int PatternIndex[MAX_PATTERN_LENGTH];		// the step number in the pattern (0 to 7)
static int PatternSwitch[MAX_PATTERN_LENGTH];		// the switch setting for the pattern step
static int PatternDwellTime[MAX_PATTERN_LENGTH];	// how long to stay on the pattern step

bool gbWantSwitch = FALSE;				// the user wants to set the switch
bool gbWantSwitch802P4T = FALSE;        // the user wants to set the switch for 802P4T
bool gbWantExtControl = FALSE;			// the user wants to set internal vs external control signals
bool gbWantSaveSettings = FALSE;		// the user wants to save the settings
bool gbWantPulseMode = FALSE;			// the user wants to start or stop pulsed mode
bool gbWantGetParam = FALSE;			// the user wants to get a parameter
bool gbWantSwitchPattern = FALSE;
bool gbGotReply = FALSE;
bool gbBatchMode = FALSE;				// the user wants to exit immediately after setting parameters in the Lab Brick
bool gbWantPulseActive = FALSE;			// the user wants to set the length of the active pulse interval
bool gbWantPulseInactive = FALSE;		// the user wants to set the length of the inactive pulse interval
bool gbWantSetPatternElement = FALSE;	// the user wants to set one or more pattern elements

// -------------------------------- Command Line routines --------------------------------
void PrintHelp()
{
	printf("Vaunix RF Switch Demo\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection. Defaults to first device ---\n");
	printf(" Each command can only appear once on the command line, except for the -v command to set pattern elements\n");
	printf("  -d nn     Select the device to work with, nn is the device number (1,2,3, etc.)\n");
	printf("  -y        Save the current settings in the device\n");
	printf("  -b        Batch mode, exit after sending device commands\n");
	printf("\n");

	printf(" --- Commands to set parameters and start patterns or pulse mode --- \n");
	printf("  -s n  Set the switch, n is the active switch number\n");
	printf("  -t n p    Set the switch, n is the active switch index, p is the port number(802P4T/802PDT)\n");
	printf("  -x n      Select internal (n=0) or external (n=1) control signals\n");
	printf("  -a nn     Set the active pulse time - RFC => RF2.\n");
	printf("  -i nn     Set the inactive pulse time - RFC => RF1.\n");
	printf("  -p n      Start pulse mode, 1 = on, 0 to stop\n");
	printf("  -v i n t	Set an element in the pattern, i is the index, 0 to 7, n is the switch setting, 1 to 4, and t is the time in seconds\n");
	printf("  -g n      Start a switch pattern, 1 = once, 2 = repeating, 0 to stop\n");

	printf("\n");

}

// --------------------- MakeLower ------------------------------

wchar_t MakeLowerW(wchar_t &wc)
{
	return wc = towlower(wc);
}

// --------------------------------------------------------------

#define MAX_MSG 32

// A function to display the status as string

wchar_t* fnLMS_perror(LVSTATUS status)
{
	wcscpy_s(errmsg, MAX_MSG, L"STATUS_OK");
	if (BAD_PARAMETER == status) wcscpy_s(errmsg, MAX_MSG, L"BAD_PARAMETER");
	if (BAD_HID_IO == status) wcscpy_s(errmsg, MAX_MSG, L"BAD_HID_IO");
	if (DEVICE_NOT_READY == status) wcscpy_s(errmsg, MAX_MSG, L"DEVICE_NOT_READY");

	// Status returns for DevStatus
	if (INVALID_DEVID == status) wcscpy_s(errmsg, MAX_MSG, L"INVALID_DEVID");
	if (DEV_CONNECTED == status) wcscpy_s(errmsg, MAX_MSG, L"DEV_CONNECTED");
	if (DEV_OPENED == status) wcscpy_s(errmsg, MAX_MSG, L"DEV_OPENED");
	if (SWPAT_ACTIVE == status) wcscpy_s(errmsg, MAX_MSG, L"PATTERN_ACTIVE");
	if (SWPAT_REPEAT == status) wcscpy_s(errmsg, MAX_MSG, L"PATTERN_REPEATING");

	return errmsg;
}

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should exit for another reason.

bool ParseCommandLine(int argc, _TCHAR *argv[])
{
	int PatIndex;
	int PatSwitch;			// temporary holders for our pattern element arguments
	float PatTime;			// we'll convert it to an integer number of milliseconds

	enum {
		wantDash, wantSwitch, wantSwitch2, wantSwitchport, wantExtControl, wantPulseMode, wantSwitchPattern, wantActive, wantInactive, wantDevID,
		wantPatternIndex, wantPatternSetting, wantPatternDwellTime
	} state = wantDash;

	for (int i = 1; i < argc; ++i)
	{
		// Convert each argument to lowercase
		wstring thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLowerW);

		if (state == wantDash)
		{
			if ('-' != thisParam[0])
			{
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}

			// remove the dash from the front of the string
			thisParam = wstring(thisParam.begin() + 1, thisParam.end());


			// Identify the command line arguments
			if (L"d" == thisParam) {
				state = wantDevID;
			}
			else if (L"b" == thisParam) {
				gbBatchMode = TRUE;
			}
			else if (L"x" == thisParam) {
				gbWantExtControl = TRUE;
				state = wantExtControl;
			}
			else if (L"s" == thisParam) {
				gbWantSwitch = TRUE;
				state = wantSwitch;
			}
			else if (L"t" == thisParam) {
				state = wantSwitch2;
			}
			else if (L"a" == thisParam) {
				gbWantPulseActive = TRUE;
				state = wantActive;
			}
			else if (L"i" == thisParam) {
				gbWantPulseInactive = TRUE;
				state = wantInactive;
			}
			else if (L"p" == thisParam) {
				gbWantPulseMode = TRUE;
				state = wantPulseMode;
			}
			else if (L"g" == thisParam) {
				gbWantSwitchPattern = TRUE;
				state = wantSwitchPattern;
			}
			else if (L"y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if (L"v" == thisParam) {
				gbWantSetPatternElement = TRUE;
				state = wantPatternIndex;
			}

			else
			{
					// this case is for "-?" and any argument we don't recognize
					PrintHelp();
					return FALSE;	// don't continue
			}
		}
		else
		{
			// save the whole substring and do conversions for each argument type
			switch (state){

			case wantSwitch:
				Switch_Select = _wtoi(thisParam.c_str());		// get the switch number (1,2,3,4)
				state = wantDash;
				break;

			case wantSwitch2:
				Switch_Select = _wtoi(thisParam.c_str());		// get the switch number (1,2,3,4)
				state = wantSwitchport;
				break;

			case wantSwitchport:
				Switch_port = _wtoi(thisParam.c_str());		// get the switch port (1,2,3,4)
				gbWantSwitch802P4T = TRUE;
				state = wantDash;
				break;

			case wantDevID:
				DevNum = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantExtControl:
				Set_External_Control = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantActive:
				ActivePulseTime = wcstof(thisParam.c_str(), NULL);
				state = wantDash;
				break;

			case wantInactive:
				InactivePulseTime = wcstof(thisParam.c_str(), NULL);
				state = wantDash;
				break;

			case wantPulseMode:
				Use_Pulse_Mode = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantSwitchPattern:
				Start_Pattern = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantPatternIndex:
				PatIndex = _wtoi(thisParam.c_str());
				if (PatIndex < 0 || PatIndex > MAX_PATTERN_LENGTH - 1)
				{
					printf("Invalid index value for pattern");
					return FALSE;
				}
				state = wantPatternSetting;
				break;

			case wantPatternSetting:
				PatSwitch = _wtoi(thisParam.c_str());
				if (PatSwitch < 1 || PatSwitch > 4)				// note - we don't know yet if this is valid for the HW we have
				{
					printf("Invalid switch value for pattern");
					return FALSE;
				}
				state = wantPatternDwellTime;
				break;

			case wantPatternDwellTime:
				PatTime = wcstof(thisParam.c_str(), NULL);			
				if (PatTime < .001 || PatTime > 1.0e+4)				// note - HW limit is larger, just a likely range check here			
				{
					printf("Out of range dwell time value for pattern");
					return FALSE;
				}
				//	now we can add the user's pattern element to our collection of them
				PatternIndex[PatIndex] = PatIndex;
				PatternSwitch[PatIndex] = PatSwitch;
				PatternDwellTime[PatIndex] = (int)PatTime * 1000;	// our units are milliseconds
				state = wantDash;
				break;



			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}



int _tmain(int argc, _TCHAR* argv[])
{
	int i, j, k, index;
	int itemp, n_entries, maxswswitches;
	bool bTemp;
	char netbuff[MAX_NETBUFF];
	float ftemp, PulseOnTime, PulseRepTime;

	printf("Lab Brick RF Switch DLL Demo Program\n");

	for (i = 0; i < MAX_MODELNAME; i++)
	{
		PatternIndex[i] = -1;			// we'll mark each entry in our list of pattern elements as untouched
	}


	if (!ParseCommandLine(argc, argv))	// the command line parser will fill in all of our variables and action flags
		return 0;

	//	 -- convert the user's device number to our internal MyDevices array index --
	//	 printf("DevNum after command line parsing = %d\n", DevNum);
	DevNum = DevNum - 1;
	if (DevNum < 0) DevNum = 0;
	if (DevNum > MAXDEVICES - 1) DevNum = MAXDEVICES - 1;

	printf("Device Selected = %d\n", DevNum + 1);

	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLSW_SetTestMode(FALSE);

	i = fnLSW_GetNumDevices();

	if (i == 0){
		printf("No device found\n");
	}

	if (i == 1){
		printf("Found %d Device\n", i);

	}
	else {
		printf("Found %d Devices\n", i);
	}

	i = fnLSW_GetDevInfo(MyDevices);

	printf("Got Device Info for %d Device[s]\n", i);


	if (i > 0)	// do we have a device?
	{
		for (j = 0; j < i; j++){

			// --- print out the first device's name ---
			itemp = fnLSW_GetModelNameA(MyDevices[j], MyDeviceName);
			printf("Device %d is an %s \n", j + 1, MyDeviceName);

			// --- print out the device's serial number ---
			itemp = fnLSW_GetSerialNumber(MyDevices[j]);
			printf("Device %d has serial number %d \n", j + 1, itemp);


			// --- We need to init the device (open it) before we can do anything else ---
			itemp = fnLSW_InitDevice(MyDevices[j]);

			if (itemp){
				printf("InitDevice returned %x\n", itemp);
			}

			// --- Lets see if we got the device's settings and parameters ---

			if(wcscmp((wchar_t *)MyDeviceName, L"LSW-802P4T"))
			{
				// Get ipmode
				itemp = fnLSW_GetIPMode(MyDevices[j]);
				printf("IP Mode:%d\n", itemp);

				// Get Ipaddress
				fnLSW_GetIPAddress(MyDevices[j],netbuff);
				printf("IP Address:%s\n", netbuff);

				// Get netmask
				fnLSW_GetNetmask(MyDevices[j],netbuff);
				printf("Subnet Mask:%s\n", netbuff);

				// Get netmask
				fnLSW_GetGateway(MyDevices[j],netbuff);
				printf("Gateway:%s\n", netbuff);

				// Get Max Switch devices
				maxswswitches = fnLSW_GetMaxSwitchDevices(MyDevices[j]);
				printf("Max Switches:%d\n", maxswswitches);

				for (index=1; index <=maxswswitches; index++)
				{
					itemp = fnLSW_GetSwitchRFoutput(MyDevices[j],index);
					printf("Switch-%d is active:%d\n", index, itemp);
				}
			}
			else if(wcscmp((wchar_t *)MyDeviceName, L"LSW-802PDT"))
			{
				// Get ipmode
				itemp = fnLSW_GetIPMode(MyDevices[j]);
				printf("IP Mode:%d\n", itemp);

				// Get Ipaddress
				fnLSW_GetIPAddress(MyDevices[j],netbuff);
				printf("IP Address:%s\n", netbuff);

				// Get netmask
				fnLSW_GetNetmask(MyDevices[j],netbuff);
				printf("Subnet Mask:%s\n", netbuff);

				// Get netmask
				fnLSW_GetGateway(MyDevices[j],netbuff);
				printf("Gateway:%s\n", netbuff);

				// Get Max Switch devices
				maxswswitches = fnLSW_GetMaxSwitchDevices(MyDevices[j]);
				printf("Max Switches:%d\n", maxswswitches);

				for (index=1; index <=maxswswitches; index++)
				{
					itemp = fnLSW_GetSwitchRFoutput(MyDevices[j],index);
					printf("Switch-%d is active:%d\n", index, itemp);
				}
			}
			else
			{

				itemp = fnLSW_GetNumSwitches(MyDevices[j]);
				printf("There are %d switches\n", itemp);

				itemp = fnLSW_GetSwitchSetting(MyDevices[j]);
				printf("Switch %d is selected\n", itemp);

				itemp = fnLSW_GetActiveSwitch(MyDevices[j]);
				printf("Switch %d is active\n", itemp);


				// --- We'll see what the switch pattern type was last set to ---

				itemp = fnLSW_GetPatternType(MyDevices[j]);
				if (!itemp){
					printf("No pattern active\n");
				}
				if (itemp & SWPAT_REPEAT){
					printf("Repeating pattern mode\n");
				}
				if (itemp & SWPAT_ONETIME){
					printf("Single pattern mode\n");
				}
				// --- and some of the other pattern values ---

				n_entries = fnLSW_GetPatternLength(MyDevices[j]);
				if (n_entries & STATUS_ERROR){
					printf("fnLSW_GetPatternLength returned error code %x\n", n_entries);
				}
				else printf("There are %d entries in the pattern\n", n_entries);

				for (k = 0; k < n_entries; k++)
				{
					itemp = fnLSW_GetPatternEntryTime(MyDevices[j], k);
					printf("Hold time for switch pattern entry %d is %d \n", k, itemp);

					itemp = fnLSW_GetPatternEntrySwitch(MyDevices[j], k);
					printf("Switch setting for switch pattern entry %d is %d \n", k, itemp);

				} // end of for loop over pattern entries


				// --- and print out some other settings from the device ---

				itemp = fnLSW_GetUseExternalControl(MyDevices[j]);
				if (itemp){
					printf("Using external control signals\n");
				}
				else {
					printf("Using internal control signals\n");
				}

				ftemp = fnLSW_GetPulseOnTime(MyDevices[j]);
				printf("Pulse On Time = %g seconds, ", ftemp);

				ftemp = fnLSW_GetPulseOffTime(MyDevices[j]);
				printf("Pulse Off Time = %g seconds\n", ftemp);

				// --- get the operation mode for pulsed operation, showing testing for errors in the return value

				itemp = fnLSW_GetPulseMode(MyDevices[j]);
				if (itemp & STATUS_ERROR){
					printf("fnLSW_GetPulseMode returned error code %x\n", itemp);
				}
				else {
					if (itemp) printf("Pulsed mode on\n");
					else printf("Pulsed mode off\n");
				}

				itemp = fnLSW_GetHasFastPulseMode(MyDevices[j]);
				if (itemp & STATUS_ERROR){
					fnLMS_perror(itemp);
					printf("fnLSW_GetHasFastPulseMode returned error code %x %s\n", itemp, errmsg);
				}
				else {
					if (itemp) printf("Pulsed mode available\n");
					else printf("Pulsed mode not available\n");
				}

			}

		} // end of the for loop over the devices


		// if the user is trying to control a device we don't have, then quit now

		if (DevNum > i - 1){

			for (j = 0; j < i; j++)
			{
				itemp = fnLSW_CloseDevice(MyDevices[j]);
			}

			return 0;			// quit - nothing else to do
		}



		// ------------- Now we'll set the requested device with new parameters -------------

		printf("Setting the Lab Brick parameters..\n");


		// --- Lets set the switch first ---

		if(wcscmp((wchar_t *)MyDeviceName, L"LSW-802P4T"))
		{
			if(gbWantSwitch802P4T)
			{
				if((Switch_Select >=1 && Switch_Select <= maxswswitches)  && (Switch_port>=1 && Switch_port <= 4))
				{
					itemp = fnLSW_SetSwitchRFoutput(MyDevices[DevNum], Switch_Select, Switch_port);
					if (!itemp) printf("Switch RF port successful...!\n");

					itemp = fnLSW_GetSwitchRFoutput(MyDevices[DevNum],Switch_Select);
					printf("Switch-%d is active:%d\n", Switch_Select, itemp);
				}
				else
				{
					printf("Arguments are out range...!\n");
				}
			}
			
		}
		else if(wcscmp((wchar_t *)MyDeviceName, L"LSW-802PDT"))
		{
			if(gbWantSwitch802P4T)
			{
				if((Switch_Select >=1 && Switch_Select <= maxswswitches)  && (Switch_port>=1 && Switch_port <= 2))
				{
					itemp = fnLSW_SetSwitchRFoutput(MyDevices[DevNum], Switch_Select, Switch_port);
					if (!itemp) printf("Switch RF port successful...!\n");

					itemp = fnLSW_GetSwitchRFoutput(MyDevices[DevNum],Switch_Select);
					printf("Switch-%d is active:%d\n", Switch_Select, itemp);
				}
				else
				{
					printf("Arguments are out range...!\n");
				}
			}
			
		}		
		else
		{

			if (gbWantSwitch)
			{
				itemp = fnLSW_SetSwitch(MyDevices[DevNum], Switch_Select);
				if (itemp) printf("fnLSW_SetSwitch returned %x\n", itemp);
			}

			// --- and then do whatever else the user requested ---

			if (gbWantExtControl)
			{
				if (Set_External_Control == 0)
				{
					bTemp = FALSE;
				}
				else
				{
					bTemp = TRUE;
				}

				itemp = fnLSW_SetUseExternalControl(MyDevices[DevNum], bTemp);
			}

			if (gbWantPulseMode)
			{
				// we'll use the values entered by the user, without error checking

				PulseOnTime = ActivePulseTime;
				PulseRepTime = ActivePulseTime + InactivePulseTime;

				// start the pulse modulation if the user wants to, otherwise stop pulse modulation
				if (Use_Pulse_Mode)
				{
					itemp = fnLSW_SetFastPulsedOutput(MyDevices[DevNum], PulseOnTime, PulseRepTime, TRUE);
					if (itemp)
					{
						fnLMS_perror(itemp);
						printf("fnLSW_SetFastPulsedOutput returned %x, %s\n", itemp, errmsg);
					}
				}
				else
				{
					itemp = fnLSW_EnableInternalPulseMod(MyDevices[DevNum], FALSE);
					if (itemp) printf("fnLSW_EnableInternalPulseMod returned %x\n", itemp);
				}
			}

			if (gbWantSwitchPattern)
			{
				// start a pattern if the user wants to, otherwise stop whatever pattern is active

				if (Start_Pattern == 0)
				{
					printf("Stopping the Switch Pattern\n");
					itemp = fnLSW_StartPattern(MyDevices[DevNum], FALSE);
					if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
				}
				else
				{
					// --- first we'll download the pattern entered by the user ---

					for (j = 0; j < MAX_PATTERN_LENGTH; j++)
					{
						if (PatternIndex[j] > 0)				// make sure the user entered a pattern element for this index
						{
							// a zero dwell time entered by the user is used to indicate the end of a pattern
							// in any event, we need an end of pattern marker at the end of the pattern, so we make sure that happens

							if (PatternDwellTime[j] == 0 || j == MAX_PATTERN_LENGTH - 1) bTemp = TRUE;
							itemp = fnLSW_SetPatternEntry(MyDevices[DevNum], PatternSwitch[j], PatternDwellTime[j], j, bTemp);
							
							if (itemp)
							{
								fnLMS_perror(itemp);
								printf("fnLSW_SetPatternEntry returned %x, %s\n", itemp, errmsg);
							}
						}

					}

					// --- and then we'll tell the hardware to execute the pattern ---

					if (Start_Pattern == 2)
					{
						itemp = fnLSW_SetPatternType(MyDevices[DevNum], TRUE);	// the user wants a repeating pattern
					}
					else
					{
						itemp = fnLSW_SetPatternType(MyDevices[DevNum], FALSE);	// the user wants the pattern to run once

					}

					printf("Starting the Switch Pattern\n");
					itemp = fnLSW_StartPattern(MyDevices[DevNum], TRUE);
					if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
				}
			}


			if (gbWantSaveSettings)
			{
				fnLSW_SaveSettings(MyDevices[DevNum]);
			}

			// -- The user wants us to exit right away --

			if (gbBatchMode)
			{
				for (j = 0; j < i; j++)
				{
					itemp = fnLSW_CloseDevice(MyDevices[j]);
				}
				return 0;		// we're done, exit to the command prompt
			}


			// -- Lets wait around some and report on the device's operation
			j = 0;

			while (j < 40)
			{

				itemp = fnLSW_GetActiveSwitch(MyDevices[DevNum]);
				printf("Switch %d is active\n", itemp);

				Sleep(500);		// wait for 1/2 second
				j++;

			}

			// -- we've done whatever the user wanted, time to close the devices

			for (j = 0; j < i; j++)
			{
				itemp = fnLSW_CloseDevice(MyDevices[j]);

			}

		}

	} // end of if ( i > 0 ) -- "we have a device"

	return 0;
}

// --------------------------- end of main ----------------------------------

